/**
  *********************************************************************************
  *
  * @file    utils.h
  * @brief   This file contains the Utilities functions/types for the driver.
  *
  * @version V1.0
  * @date    07 Nov 2017
  * @author  AE Team
  * @note
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  */

#ifndef __UTILS_H__
#define __UTILS_H__

#ifdef __cplusplus
 extern "C" {
#endif

#include <stdlib.h>
#include "ald_conf.h"
#include "type.h"
#include "es32f0xx.h"


/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @addtogroup UTILS
  * @{
  */

/** @defgroup ALD_Public_Types Public Types
  * @{
  */
/**
  * @brief  SysTick interval
  */
extern uint32_t __systick_interval;

/**
  * @brief  ALD Status structures definition
  */
typedef enum {
	OK      = 0x0,
	ERROR   = 0x1,
	BUSY    = 0x2,
	TIMEOUT = 0x3
} ald_status_t;

/**
  * @brief  SysTick interval definition
  */
typedef enum {
	SYSTICK_INTERVAL_1MS    = 1000,	/**< Interval is 1ms */
	SYSTICK_INTERVAL_10MS   = 100,	/**< Interval is 10ms */
	SYSTICK_INTERVAL_100MS  = 10,	/**< Interval is 100ms */
	SYSTICK_INTERVAL_1000MS = 1,	/**< Interval is 1s */
} systick_interval_t;
/**
  * @}
  */

/** @defgroup ALD_Public_Macros Public Macros
  * @{
  */
#define ALD_MAX_DELAY	0xFFFFFFFF

#define IS_BIT_SET(reg, bit)	(((reg) & (bit)) != RESET)
#define IS_BIT_CLR(reg, bit)	(((reg) & (bit)) == RESET)
#define RESET_HANDLE_STATE(x)	((x)->state = 0)
#define __LOCK(x)				\
	do {					\
		if ((x)->lock == LOCK) {	\
			return BUSY;		\
		}				\
		else {				\
			(x)->lock = LOCK;	\
		}				\
	} while (0)

#define __UNLOCK(x)				\
	do {					\
		(x)->lock = UNLOCK;		\
	} while (0)

/**
  * @}
  */

/** @defgroup ALD_Private_Macros Private Macros
  * @{
  */
#define IS_PRIO(x)	((x) < 4)
#define IS_SYSTICK_INTERVAL(x)	(((x) == SYSTICK_INTERVAL_1MS)   || \
                                 ((x) == SYSTICK_INTERVAL_10MS)  || \
                                 ((x) == SYSTICK_INTERVAL_100MS) || \
                                 ((x) == SYSTICK_INTERVAL_1000MS))
/**
  * @}
  */

/** @addtogroup ALD_Public_Functions
  * @{
  */

/** @addtogroup ALD_Public_Functions_Group1
  * @{
  */

/* Initialization functions */
void mcu_ald_init(void);
void __init_tick(uint32_t prio);
void systick_interval_select(systick_interval_t value);

/**
  * @}
  */

/** @addtogroup ALD_Public_Functions_Group2
  * @{
  */
/* Peripheral Control functions */
void __inc_tick(void);
void __delay_ms(__IO uint32_t delay);
uint32_t __get_tick(void);
void __suspend_tick(void);
void __resume_tick(void);
void systick_irq_cbk(void);
uint32_t get_ald_version(void);
ald_status_t __wait_flag(uint32_t *reg, uint32_t bit, flag_status_t status, uint32_t timeout);
void mcu_irq_config(IRQn_Type irq, uint8_t prio, type_func_t status);
uint32_t mcu_get_tick(void);
uint32_t mcu_get_cpu_id(void);

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif

#endif /* __UTILS_H__ */
