/**
  *********************************************************************************
  *
  * @file    hal_crc.c
  * @brief   CRC module driver.
  *
  * @version V1.0
  * @date    6 Dec 2017
  * @author  AE Team
  * @note
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  */

#include "hal_crc.h"

/** @addtogroup ES32FXXX_HAL
  * @{
  */

/** @defgroup CRC CRC
  * @brief CRC module driver
  * @{
  */
#ifdef HAL_CRC

/** @addtogroup CRC_Private_Functions   CRC Private Functions
  * @{
  */
void crc_reset(crc_handle_t *hperh);
#ifdef HAL_DMA
static void crc_dma_calculate_cplt(void *arg);
static void crc_dma_error(void *arg);
#endif
/**
  * @}
  */


/** @defgroup CRC_Public_Functions CRC Public Functions
  * @{
  */

/** @defgroup CRC_Public_Functions_Group1 Initialization functions
  * @brief Initialization and Configuration functions
  * @{
  */

/**
  * @brief  Initializes the CRC mode according to the specified parameters in
  *         the SPI_init_t and create the associated handle.
  * @param  hperh: Pointer to a crc_handle_t structure that contains
  *         the configuration information for the specified CRC module.
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t crc_init(crc_handle_t *hperh)
{
	if (hperh == NULL)
		return ERROR;

	assert_param(IS_CRC(hperh->perh));
	assert_param(IS_CRC_MODE(hperh->init.mode));
	assert_param(IS_FUNC_STATE(hperh->init.chs_rev_en));
	assert_param(IS_FUNC_STATE(hperh->init.data_inv_en));
	assert_param(IS_FUNC_STATE(hperh->init.data_rev_en));
	assert_param(IS_FUNC_STATE(hperh->init.chs_inv_en));

	crc_reset(hperh);
	__LOCK(hperh);

	if (hperh->state != CRC_STATE_RESET) {
		__UNLOCK(hperh);
		return ERROR;
	}

	CRC_ENABLE(hperh);

	hperh->perh->CR.CHSREV = hperh->init.chs_rev_en;
	hperh->perh->CR.DATINV = hperh->init.data_inv_en;
	hperh->perh->CR.CHSINV = hperh->init.chs_inv_en;
	hperh->perh->CR.MODE   = hperh->init.mode;
	hperh->perh->CR.DATLEN = CRC_DATASIZE_8;
	hperh->perh->CR.DATREV = hperh->init.data_rev_en;
	hperh->perh->CR.BYTORD = 1;

	hperh->perh->SEED.SEED = hperh->init.seed;
	CRC_RESET(hperh);

	hperh->state           = CRC_STATE_READY;

	__UNLOCK(hperh);
	return OK;
}

/**
  * @}
  */

/** @defgroup CRC_Public_Functions_Group2 Calculate functions
  * @brief Calculate functions
  * @{
  */

/**
  * @brief  Calculate the crc value of data.
  * @param  hperh: Pointer to a crc_handle_t structure that contains
  *         the configuration information for the specified CRC module.
  * @param  buf: Pointer to data buffer
  * @param  size: The size of data to be calculate
  * @retval result, the result of a amount data
  */
uint32_t crc_calculate(crc_handle_t *hperh, uint8_t *buf, uint32_t size)
{
	uint32_t i;
	uint32_t ret;

	assert_param(IS_CRC(hperh->perh));

	if (buf == NULL||size == 0)
		return 0;

	__LOCK(hperh);
	hperh->state = CRC_STATE_BUSY;

	for (i = 0;i < size; i++) {
 		CRC->DATA.Byte[0] = *(buf + i);
 	}

	ret          = CRC->CHECKSUM.CHECKSUM;
	hperh->state = CRC_STATE_READY;
	__UNLOCK(hperh);

	return ret;
}
/**
  * @}
  */

#ifdef HAL_DMA
/** @defgroup CRC_Public_Functions_Group3 DMA operation functions
  * @brief DMA operation functions
  * @{
  */

/**
  * @brief  Calculate an amount of data used dma channel
  * @param  hperh: Pointer to a crc_handle_t structure that contains
  *         the configuration information for the specified CRC module.
  * @param  buf: Pointer to data buffer
  * @param  res: Pointer to result
  * @param  size: Amount of data to be Calculate
  * @param  channel: DMA channel as CRC transmit
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t crc_calculate_by_dma(crc_handle_t *hperh, uint8_t *buf, uint32_t *res, uint16_t size, uint8_t channel)
{
	if (hperh->state != CRC_STATE_READY)
		return BUSY;

	if (buf == NULL|| size == 0)
		return ERROR;

	__LOCK(hperh);

	hperh->state = CRC_STATE_BUSY;

	hperh->cal_buf = buf;
	hperh->cal_res = res;

	if (hperh->hdma.perh == NULL)
		hperh->hdma.perh = DMA0;

	hperh->hdma.cplt_arg = (void *)hperh;
	hperh->hdma.cplt_cbk = &crc_dma_calculate_cplt;
	hperh->hdma.err_arg  = (void *)hperh;
	hperh->hdma.err_cbk  = &crc_dma_error;

	dma_config_struct(&(hperh->hdma.config));
	hperh->hdma.config.data_width = DMA_DATA_SIZE_BYTE;
	hperh->hdma.config.src        = (void *)buf;
	hperh->hdma.config.dst        = (void *)&hperh->perh->DATA.Word;
	hperh->hdma.config.size       = size;
	hperh->hdma.config.src_inc    = DMA_DATA_INC_BYTE;
	hperh->hdma.config.dst_inc    = DMA_DATA_INC_NONE;
	hperh->hdma.config.msel       = DMA_MSEL_CRC;
	hperh->hdma.config.msigsel    = DMA_MSIGSEL_NONE;
	hperh->hdma.config.channel    = channel;
	dma_config_basic(&(hperh->hdma));

	__UNLOCK(hperh);
	CRC_DMA_ENABLE(hperh);

	return OK;
}

/**
  * @brief  Pauses the DMA Transfer.
  * @param  hperh: Pointer to a crc_handle_t structure that contains
  *         the configuration information for the specified CRC module.
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t crc_dma_pause(crc_handle_t *hperh)
{
	__LOCK(hperh);
	CRC_DMA_DISABLE(hperh);
	__UNLOCK(hperh);

	return OK;
}

/**
  * @brief  Resumes the DMA Transfer.
  * @param  hperh: Pointer to a crc_handle_t structure that contains
  *         the configuration information for the specified CRC module.
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t crc_dma_resume(crc_handle_t *hperh)
{
	__LOCK(hperh);
	CRC_DMA_ENABLE(hperh);
	__UNLOCK(hperh);

	return OK;
}

/**
  * @brief  Stops the DMA Transfer.
  * @param  hperh: Pointer to a crc_handle_t structure that contains
  *         the configuration information for the specified CRC module.
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t crc_dma_stop(crc_handle_t *hperh)
{
	__LOCK(hperh);
	CRC_DMA_DISABLE(hperh);
	__UNLOCK(hperh);

	hperh->state = CRC_STATE_READY;
	return OK;
}

/**
  * @}
  */
#endif

/** @defgroup CRC_Public_Functions_Group4 Peripheral State and Errors functions
  * @brief    CRC State and Errors functions
  * @{
  */

/**
  * @brief  Returns the CRC state.
  * @param  hperh: Pointer to a crc_handle_t structure that contains
  *         the configuration information for the specified CRC module.
  * @retval CRC state
  */
crc_state_t crc_get_state(crc_handle_t *hperh)
{
	assert_param(IS_CRC(hperh->perh));

	return hperh->state;
}
/**
  * @}
  */

/**
  * @}
  */

/** @defgroup CRC_Private_Functions   CRC Private Functions
  *  @brief   CRC Private functions
  * @{
  */

/**
  * @brief  Reset the CRC peripheral.
  * @param  hperh: Pointer to a crc_handle_t structure that contains
  *         the configuration information for the specified CRC module.
  * @retval None
  */
void crc_reset(crc_handle_t *hperh)
{
	hperh->perh->DATA.Word = 0x0;
	hperh->perh->CR.Word   = 0x2;
	hperh->perh->SEED.Word = 0xFFFFFFFF;

	hperh->state = CRC_STATE_READY;
	__UNLOCK(hperh);
	return;
}

#ifdef HAL_DMA
/**
  * @brief  DMA CRC calculate process complete callback.
  * @param  arg: Pointer to a crc_handle_t structure that contains
  *         the configuration information for the specified CRC module.
  * @retval None
  */
static void crc_dma_calculate_cplt(void *arg)
{
	crc_handle_t *hperh = (crc_handle_t *)arg;

	*(hperh->cal_res) = CRC->CHECKSUM.CHECKSUM;
	CRC_DMA_DISABLE(hperh);

	hperh->state = CRC_STATE_READY;

	if (hperh->cal_cplt_cbk)
		hperh->cal_cplt_cbk(hperh);
}

/**
  * @brief  DMA CRC communication error callback.
  * @param  arg: Pointer to a crc_handle_t structure that contains
  *         the configuration information for the specified CRC module.
  * @retval None
  */
static void crc_dma_error(void *arg)
{
	crc_handle_t *hperh = (crc_handle_t *)arg;

	CRC_CLEAR_ERROR_FLAG(hperh);
	CRC_DMA_DISABLE(hperh);

	hperh->state = CRC_STATE_READY;

	if (hperh->err_cplt_cbk)
		hperh->err_cplt_cbk(hperh);
}
#endif
/**
  * @}
  */

/**
  * @}
  */
#endif /* HAL_CRC */

/**
  * @}
  */

