/**
  *********************************************************************************
  *
  * @file    hal_pmu.c
  * @brief   PMU module driver.
  *
  * @version V1.0
  * @date    04 Dec 2017
  * @author  AE Team
  * @note
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  */

#include "hal_pmu.h"


/** @addtogroup ES32FXXX_HAL
  * @{
  */

/** @defgroup PMU PMU
  * @brief PMU module driver
  * @{
  */
#ifdef HAL_PMU


/** @defgroup PMU_Private_Functions PMU Private Functions
  * @{
  */

/**
  * @brief  PMU module interrupt handler
  * @retval None
  */
void LVD_Handler(void)
{
	SYSCFG_UNLOCK();
	PMU->LVDCR.LVDCIF = 1;
	SYSCFG_LOCK();

	lvd_irq_cbk();
	return;
}
/**
  * @}
  */

/** @defgroup PMU_Public_Functions PMU Public Functions
  * @{
  */

/** @addtogroup PMU_Public_Functions_Group1 Low Power Mode
  * @brief Low power mode select functions
  *
  * @verbatim
  ==============================================================================
              ##### Low power mode select functions #####
  ==============================================================================
    [..]  This section provides functions allowing to:
      (+) Enter stop1 mode.
      (+) Enter stop2 mode.
      (+) Enter standby mode.

    @endverbatim
  * @{
  */

/**
  * @brief  Enter stop1 mode
  * @retval None
  */
void pmu_stop1_enter(void)
{
	SYSCFG_UNLOCK();
	PMU->CR.LPM = PMU_LP_STOP1;
	SYSCFG_LOCK();

	__sleep_deep();
	return;
}

/**
  * @brief  Enter stop2 mode
  * @retval None
  */
void pmu_stop2_enter(void)
{
	SYSCFG_UNLOCK();
	PMU->CR.LPM = PMU_LP_STOP2;
	SYSCFG_LOCK();

	__sleep_deep();
	return;

}



/**
  * @brief  Enter standby mode
  * @param  port: The port whick wake up the standby mode.
  * @retval None
  */
void pmu_standby_enter(pmu_standby_wakeup_sel_t port)
{
	assert_param(IS_PMU_STANDBY_PORT_SEL(port));
	SYSCFG_UNLOCK();

	PMU->CR.WKPS  = port;
	PMU->CR.WKPEN = ENABLE;
	PMU->CR.LPM   = PMU_LP_STANDBY;

	SYSCFG_LOCK();
	__sleep_deep();

	return;
}
/**
  * @}
  */

/** @addtogroup PMU_Public_Functions_Group2 LVD Configure
  * @brief LVD configure functions
  *
  * @verbatim
  ==============================================================================
              ##### LVD configure functions #####
  ==============================================================================
    [..]  This section provides functions allowing to:
      (+) Configure lvd parameters.
      (+) Interrupt callback function.

    @endverbatim
  * @{
  */

/**
  * @brief  Configure lvd using specified parameters.
  * @param  sel: LVD threshold voltage.
  * @param  mode: LVD trigger mode.
  * @param  state: New state, ENABLE/DISABLE;
  * @retval None
  */
void pmu_lvd_config(pmu_lvd_voltage_sel_t sel, pmu_lvd_trigger_mode_t mode, type_func_t state)
{
	assert_param(IS_FUNC_STATE(state));
	SYSCFG_UNLOCK();

	if (state) {
		assert_param(IS_PMU_LVD_VOL_SEL(sel));
		assert_param(IS_PMU_LVD_TRIGGER_MODE(mode));

		PMU->LVDCR.LVDS   = sel;
		PMU->LVDCR.LVIFS  = mode;
		PMU->LVDCR.LVDFLT = 1;
		PMU->LVDCR.LVDCIF = 1;
		PMU->LVDCR.LVDIE  = 1;
		PMU->LVDCR.LVDEN  = 1;
	}
	else {
		PMU->LVDCR.LVDCIF = 1;
		PMU->LVDCR.LVDIE  = 0;
		PMU->LVDCR.LVDEN  = 0;
	}

	SYSCFG_LOCK();
	return;
}

/**
  * @brief  Interrupt callback function.
  * @note   This function is declared as __weak to be overwritten in case of other
  *         implementations in user file.
  * @retval None
  */
__weak void lvd_irq_cbk(void)
{
	return;
}
/**
  * @}
  */


/**
  * @}
  */
#endif /* HAL_PMU */
/**
  * @}
  */

/**
  * @}
  */
