	/**
  *********************************************************************************
  *
  * @file    hal_spi.c
  * @brief   SPI module driver.
  *          This file provides firmware functions to manage the following
  *          functionalities of SPI peripheral:
  *           + Initialization functions
  *           + IO operation functions
  *           + Peripheral Control functions
  *           + Peripheral State functions
  *
  * @version V1.0
  * @date    13 Nov 2017
  * @author  AE Team
  * @note
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  @verbatim
  ==============================================================================
                        ##### How to use this driver #####
  ==============================================================================
    [..]
      The SPI driver can be used as follows:

      (#) Declare a spi_handle_t structure, for example:
          spi_handle_t hperh;

      (#) Initialize the SPI low level resources:
          (##) Enable the SPIx interface clock
          (##) SPI pins configuration
              (+++) Enable the clock for the SPI GPIOs
              (+++) Configure these SPI pins as push-pull
          (##) NVIC configuration if you need to use interrupt process
               by implementing the mcu_irq_config() API.
	       Invoked spi_irq_handle() function in SPI-IRQ function
          (##) DMA Configuration if you need to use DMA process
              (+++) Define HAL_DMA in hal_conf.h
 	      (+++) Enable the DMAx clock

      (#) Program the Mode, Direction , Data size, Baudrate Prescaler, NSS
          management, Clock polarity and phase, FirstBit and CRC configuration in the hspi Init structure.

      (#) Initialize the SPI module by invoking the spi_init() API.

     [..]
       Circular mode restriction:
      (#) The DMA circular mode cannot be used when the SPI is configured in these modes:
          (##) Master 2Lines RxOnly
          (##) Master 1Line Rx
      (#) When the SPI DMA Pause/Stop features are used, we must use the following APIs
          the spi_dma_pause()/ spi_dma_stop().

  * @endverbatim
  */

#include "hal_spi.h"


/** @addtogroup ES32FXXX_HAL
  * @{
  */

/** @defgroup SPI SPI
  * @brief SPI module driver
  * @{
  */
#ifdef HAL_SPI

/** @addtogroup SPI_Private_Functions   SPI Private Functions
  * @{
  */
static hal_status_t spi_wait_flag(spi_handle_t *hperh, spi_flag_t flag, flag_status_t status, uint32_t timeout);
static void __spi_send_by_it(spi_handle_t *hperh);
static void __spi_recv_by_it(spi_handle_t *hperh);
static void __spi_send_recv_by_it(spi_handle_t *hperh);
#ifdef HAL_DMA
static void spi_dma_send_cplt(void *arg);
static void spi_dma_recv_cplt(void *arg);
static void spi_dma_send_recv_cplt(void *arg);
static void spi_dma_error(void *arg);
#endif
/**
  * @}
  */

/** @defgroup SPI_Public_Functions SPI Public Functions
  * @{
  */

/** @defgroup SPI_Public_Functions_Group1 Initialization functions
  * @brief Initialization and Configuration functions
  *
  * @verbatim
 ===============================================================================
              ##### Initialization and de-initialization functions #####
 ===============================================================================
    [..]  This subsection provides a set of functions allowing to initialize and
          reset the SPIx peripheral:

      (+) User must configure all related peripherals resources
          (CLOCK, GPIO, DMA, NVIC).

      (+) Call the function spi_init() to configure the selected device with
          the selected configuration:
        (++) Mode
        (++) Direction
        (++) Data Size
        (++) Clock Polarity and Phase
        (++) NSS Management
        (++) BaudRate Prescaler
        (++) FirstBit
        (++) TIMode
        (++) CRC Calculation
        (++) CRC Polynomial if CRC enabled

      (+) Call the function spi_reset() to reset the selected SPIx periperal.

    @endverbatim
  * @{
  */

/**
  * @brief  Reset the SPI peripheral.
  * @param  hperh: Pointer to a spi_handle_t structure that contains
  *         the configuration information for the specified SPI module.
  * @retval None
  */
void spi_reset(spi_handle_t *hperh)
{
	hperh->perh->CR1.Word   = 0x0;
	hperh->perh->SR.Word    = 0x2;
	hperh->perh->CRCPR.Word = 0x7;

	SPI_RESET_HANDLE_STATE(hperh);
	__UNLOCK(hperh);

	return;
}

/**
  * @brief  Initializes the SPI mode according to the specified parameters in
  *         the SPI_init_t and create the associated handle.
  * @param  hperh: Pointer to a spi_handle_t structure that contains
  *         the configuration information for the specified SPI module.
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t spi_init(spi_handle_t *hperh)
{
	if (hperh == NULL)
		return ERROR;

	assert_param(IS_SPI(hperh->perh));
	assert_param(IS_SPI_MODE(hperh->init.mode));
	assert_param(IS_SPI_DIRECTION(hperh->init.dir));
	assert_param(IS_SPI_BR(hperh->init.psc));
	assert_param(IS_FUNC_STATE(hperh->init.first_bit));
	assert_param(IS_FUNC_STATE(hperh->init.ssm_en));
	assert_param(IS_FUNC_STATE(hperh->init.crc_calc));
	assert_param(IS_SPI_DATASIZE(hperh->init.data_size));
	assert_param(IS_SPI_CPHA(hperh->init.phase));
	assert_param(IS_SPI_CPOL(hperh->init.polarity));
	assert_param(IS_SPI_FRAME(hperh->init.frame));

	spi_reset(hperh);

	hperh->perh->CR1.CPHA     = hperh->init.phase;
	hperh->perh->CR1.CPOL     = hperh->init.polarity;
	hperh->perh->CR1.BR       = hperh->init.psc;
	hperh->perh->CR2.DS       = hperh->init.data_size;
	hperh->perh->CR2.FRF      = hperh->init.frame;
	hperh->perh->CR1.MSTR     = hperh->init.mode;
	hperh->perh->CR1.SSI      = hperh->init.mode == SPI_MODE_MASTER ? 1 : 0;
	hperh->perh->CR1.SSM      = hperh->init.ssm_en;
	hperh->perh->CR1.LSBFIRST = hperh->init.first_bit;

	if (hperh->init.dir == SPI_DIRECTION_2LINES) {
		hperh->perh->CR1.BIDIMODE = 0;
		hperh->perh->CR1.RXONLY   = 0;
	}
	else if (hperh->init.dir == SPI_DIRECTION_2LINES_RXONLY) {
		hperh->perh->CR1.BIDIMODE = 0;
		hperh->perh->CR1.RXONLY   = 1;
	}
	else if (hperh->init.dir == SPI_DIRECTION_1LINE_RX) {
		hperh->perh->CR1.BIDIMODE = 1;
		hperh->perh->CR1.BIDIOE   = 0; 
	}
	else {
		hperh->perh->CR1.BIDIMODE = 1;
		hperh->perh->CR1.BIDIOE   = 1; 
	}

	/* configure CRC */
	hperh->perh->CR1.CRCEN     = hperh->init.crc_calc;
	hperh->perh->CR1.CRCL      = hperh->init.crc_len;
	hperh->perh->CRCPR.CRCPOLY = hperh->init.crc_poly;

	hperh->err_code = SPI_ERROR_NONE;
	hperh->state    = SPI_STATE_READY;

	return OK;
}

/**
  * @brief  Config the fifo threshold 
  * @param  hperh: Pointer to a spi_handle_t structure that contains
  *         the configuration information for the specified SPI module.
  * @param  threshold: The threshold value of fifo. 
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t spi_fifo_threshold_config(spi_handle_t *hperh, uint8_t threshold)
{
	assert_param(IS_SPI(hperh->perh));
	assert_param(IS_SPI_FIFO_THRESHOLD(threshold));

	hperh->perh->CR2.RXFTH = threshold;
	hperh->perh->CR2.TXFTH = threshold;

	return OK;
}

/**
  * @brief  Select the data transfer direction in bidrectional.
  * @param  hperh: Pointer to a spi_handle_t structure that contains
  *         the configuration information for the specified SPI module.
  * @param  direction: The transfer direction in bidirectional mode.
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t spi_bidirection_line_config(spi_handle_t *hperh, spi_bidi_t direction)
{
	assert_param(IS_SPI(hperh->perh));
	assert_param(IS_SPI_BIDIOE(direction));

	hperh->perh->CR1.BIDIOE = direction;

	return OK;
}
/**
  * @}
  */

/** @defgroup SPI_Public_Functions_Group2 IO operation functions
  * @brief SPI Transmit and Receive functions
  *
  * @verbatim
  ==============================================================================
                      ##### IO operation functions #####
 ===============================================================================
    This subsection provides a set of functions allowing to manage the SPI
    data transfers.

    [..] The SPI supports master or slave mode:

    (#) There are two modes of transfer:
       (++) Blocking mode: The communication is performed in polling mode.
            The HAL status of all data processing is returned by the same function
            after finishing transfer.
       (++) No-Blocking mode: The communication is performed using Interrupts
            or DMA, These APIs return the HAL status.
            The end of the data processing will be indicated through the
            dedicated SPI IRQ when using Interrupt mode or the DMA IRQ when
            using DMA mode.
            The hperh->tx_cplt_cbk(), hperh->rx_cplt_cbk() and hperh->tx_rx_cplt_cbk() user callbacks
            will be executed respectivelly at the end of the transmit or Receive process
            The hperh->err_cbk() user callback will be executed when a communication error is detected

    (#) APIs provided for these 2 transfer modes (Blocking mode or Non blocking mode using either Interrupt or DMA)
        exist for 1Line (simplex) and 2Lines (full duplex) modes.

  * @endverbatim
  * @{
  */

/**
  * @brief  transmit an amount of data in blocking mode.
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @param  timeout: Timeout duration
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t spi_send(spi_handle_t *hperh, uint8_t *buf, uint16_t size, uint32_t timeout)
{
	assert_param(IS_SPI(hperh->perh));

	if (hperh->state != SPI_STATE_READY)
		return BUSY;

	if (buf == NULL || size == 0)
		return ERROR;

	__LOCK(hperh);

	hperh->state    = SPI_STATE_BUSY_TX;
	hperh->err_code = SPI_ERROR_NONE;

	hperh->tx_buf   = buf;
	hperh->tx_size  = size;
	hperh->tx_count = size;
	hperh->rx_buf   = NULL;
	hperh->rx_size  = 0;
	hperh->rx_count = 0;

	if (hperh->init.crc_calc)
		SPI_CRC_RESET(hperh);
	if (hperh->init.dir == SPI_DIRECTION_1LINE_TX)
		SPI_1LINE_TX(hperh);
	if (hperh->perh->CR1.SPE != ENABLE)
		SPI_ENABLE(hperh);

	while (hperh->tx_count > 0) {
		if (spi_wait_flag(hperh, SPI_FLAG_TXE, SET, timeout) != OK) {
			SPI_DISABLE(hperh);

			if (hperh->init.crc_calc)
				SPI_CRC_RESET(hperh);

			hperh->state = SPI_STATE_READY;
			__UNLOCK(hperh);
			return TIMEOUT;
		}

		if (hperh->init.data_size <= SPI_DATA_SIZE_8) {
			*((volatile uint8_t *)hperh->perh + 0x0c) = *hperh->tx_buf;
			++hperh->tx_buf;
			--hperh->tx_count;
		}
		else {
			hperh->perh->DR.DR = (*(uint16_t *)hperh->tx_buf);
			hperh->tx_buf +=2;
			--hperh->tx_count;
		}
	}

	if (hperh->init.crc_calc)
		SPI_CRCNEXT_ENABLE(hperh);

 	if ((spi_wait_flag(hperh, SPI_FLAG_TXE, SET, timeout) != OK)
			|| (spi_wait_flag(hperh, SPI_FLAG_BSY, RESET, timeout) != OK)) {
 		SPI_DISABLE(hperh);

 		if (hperh->init.crc_calc)
 			SPI_CRC_RESET(hperh);

 		hperh->state = SPI_STATE_READY;
 		__UNLOCK(hperh);
 		return TIMEOUT;
 	}

	hperh->state = SPI_STATE_READY;
	__UNLOCK(hperh);

	return OK;
}

/**
  * @brief  Receive an amount of data in blocking mode.
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be received
  * @param  timeout: Timeout duration
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t spi_recv(spi_handle_t *hperh, uint8_t *buf, uint16_t size, uint32_t timeout)
{
	assert_param(IS_SPI(hperh->perh));

	if(hperh->state != SPI_STATE_READY)
		return BUSY;
	if(buf == NULL || size == 0)
		return ERROR;

	__LOCK(hperh);
	hperh->state    = SPI_STATE_BUSY_RX;
	hperh->err_code = SPI_ERROR_NONE;

	hperh->rx_buf   = buf;
	hperh->rx_size  = size;
	hperh->rx_count = size;
	hperh->tx_buf   = NULL;
	hperh->tx_size  = 0;
	hperh->tx_count = 0;

	if (hperh->init.dir == SPI_DIRECTION_1LINE_RX)
		SPI_1LINE_RX(hperh);

	if ((hperh->init.mode == SPI_MODE_MASTER) && (hperh->init.dir == SPI_DIRECTION_2LINES)) {
		hperh->state    = SPI_STATE_READY;
		__UNLOCK(hperh);
		return spi_send_recv(hperh, buf, buf, size, timeout);
	}

	if (hperh->perh->CR1.SPE != ENABLE)
		SPI_ENABLE(hperh);

	if((hperh->init.crc_calc == ENABLE) && (hperh->rx_count > 0)) {
		if (hperh->init.data_size <= SPI_DATA_SIZE_8) {
			*((uint8_t *)&(hperh->perh->DR)) = 0xB1;
		}
		else {
			*((uint16_t *)&(hperh->perh->DR)) = 0xB2B1;
		}
	}

	while (hperh->rx_count > 0) {
		if ((hperh->rx_count > 1) && (hperh->init.crc_calc == ENABLE)) {
			if (hperh->init.data_size <= SPI_DATA_SIZE_8) {
				*((uint8_t *)&(hperh->perh->DR)) = 0xB1;
			}
			else {
				*((uint16_t *)&(hperh->perh->DR)) = 0xB2B1;
			}
		}

		if (spi_wait_flag(hperh, SPI_FLAG_RXE, RESET, timeout) != OK) {
			SPI_DISABLE(hperh);

			if(hperh->init.crc_calc)
				SPI_CRC_RESET(hperh);

			hperh->state = SPI_STATE_READY;
			__UNLOCK(hperh);
			return TIMEOUT;
		}

		if ((hperh->init.crc_calc == ENABLE) && (hperh->perh->CR1.CRCNEXT == 0)) {
			spi_interrupt_config(hperh, SPI_IT_CRCERR, ENABLE);
			hperh->perh->CR1.CRCNEXT = 1;	
		}

		if (hperh->init.data_size <= SPI_DATA_SIZE_8) {
			*hperh->rx_buf = hperh->perh->DR.DR;
			++hperh->rx_buf;
			--hperh->rx_count;
		}
		else {
			*(uint16_t *)hperh->rx_buf = hperh->perh->DR.DR;
			hperh->rx_buf +=2;
			--hperh->rx_count;
		}
	}

	if (hperh->init.crc_calc == ENABLE) {
		if (spi_wait_flag(hperh, SPI_FLAG_RXE, RESET, timeout) != OK) {
			SPI_DISABLE(hperh);

			if(hperh->init.crc_calc)
				SPI_CRC_RESET(hperh);

			hperh->state = SPI_STATE_READY;
			__UNLOCK(hperh);
			return TIMEOUT;
		}

		if (hperh->init.data_size <= SPI_DATA_SIZE_8) {
			*hperh->rx_buf = hperh->perh->DR.DR;
			++hperh->rx_buf;
			--hperh->rx_count;
		}
		else {
			*(uint16_t *)hperh->rx_buf = hperh->perh->DR.DR;
			hperh->rx_buf +=2;
			--hperh->rx_count;
		}
	}

	if ((hperh->init.mode == SPI_MODE_MASTER) && (hperh->init.dir != SPI_DIRECTION_2LINES))
		SPI_DISABLE(hperh);


	hperh->state = SPI_STATE_READY;
	__UNLOCK(hperh);

	return OK;
}

/**
  * @brief  Full-Duplex Send receive an amount of data in full-duplex mode (blocking mode).
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @param  tx_buf: Pointer to data transmitted buffer
  * @param  rx_buf: Pointer to data received buffer
  * @param  size: Amount of data to be sent
  * @param  timeout: Timeout duration
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t spi_send_recv(spi_handle_t *hperh, uint8_t *tx_buf, uint8_t *rx_buf, uint16_t size, uint32_t timeout)
{
	uint16_t temp;

	assert_param(IS_SPI(hperh->perh));

	if (hperh->state != SPI_STATE_READY)
		return BUSY;
	if (hperh->init.dir != SPI_DIRECTION_2LINES)
		return ERROR;
	if (tx_buf == NULL || rx_buf == NULL || size == 0)
		return ERROR;

	__LOCK(hperh);
	hperh->state    = SPI_STATE_BUSY_TX_RX;
	hperh->err_code = SPI_ERROR_NONE;

	hperh->tx_buf   = tx_buf;
	hperh->tx_size  = size;
	hperh->tx_count = size;
	hperh->rx_buf   = rx_buf;
	hperh->rx_size  = size;
	hperh->rx_count = size;

	if (hperh->init.crc_calc)
		SPI_CRC_RESET(hperh);
	if (hperh->perh->CR1.SPE != ENABLE)
		SPI_ENABLE(hperh);

	while (hperh->tx_count > 0) {
		if (spi_wait_flag(hperh, SPI_FLAG_TXE, SET, timeout) != OK) {
			SPI_DISABLE(hperh);

			if (hperh->init.crc_calc)
				SPI_CRC_RESET(hperh);

			hperh->state = SPI_STATE_READY;
			__UNLOCK(hperh);
			return TIMEOUT;
		}

		if (hperh->init.data_size <= SPI_DATA_SIZE_8) {
			*((volatile uint8_t *)hperh->perh + 0x0c) = *hperh->tx_buf;
			++hperh->tx_buf;
			--hperh->tx_count;
		}
		else {
			hperh->perh->DR.DR = (*(uint16_t *)hperh->tx_buf);
			hperh->tx_buf += 2;
			--hperh->tx_count;
		}

		if (spi_wait_flag(hperh, SPI_FLAG_RXE, RESET, timeout) != OK) {
			SPI_DISABLE(hperh);

			if (hperh->init.crc_calc)
				SPI_CRC_RESET(hperh);

			hperh->state = SPI_STATE_READY;
			__UNLOCK(hperh);
			return TIMEOUT;
		}

		if (hperh->init.data_size <= SPI_DATA_SIZE_8) {
			*hperh->rx_buf = hperh->perh->DR.DR;
			++hperh->rx_buf;
			--hperh->rx_count;
		}
		else {
			(*(uint16_t*)hperh->tx_buf) = hperh->perh->DR.DR;
			hperh->rx_buf += 2;
			--hperh->rx_count;
		}
	}

	if (hperh->init.crc_calc) {
		SPI_CRCNEXT_ENABLE(hperh);

		if (spi_wait_flag(hperh, SPI_FLAG_TXE, SET, timeout) != OK) {
			SPI_DISABLE(hperh);

			if (hperh->init.crc_calc)
				SPI_CRC_RESET(hperh);

			hperh->state = SPI_STATE_READY;
			__UNLOCK(hperh);
			return TIMEOUT;
		}
		if ((spi_wait_flag(hperh, SPI_FLAG_RXE, RESET, timeout) != OK)) {
			SPI_DISABLE(hperh);

			if (hperh->init.crc_calc)
				SPI_CRC_RESET(hperh);

			hperh->state = SPI_STATE_READY;
			__UNLOCK(hperh);
			return TIMEOUT;
		}

		temp = hperh->perh->DR.DR;
		UNUSED(temp);
	}


	hperh->state = SPI_STATE_READY;
	__UNLOCK(hperh);

	return OK;
}

/**
  * @brief  Wraps up transmission in non blocking mode.
  * @param  hperh: pointer to a spi_handle_t structure.
  * @param  buf: Pointer to data transmitted buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t spi_send_by_it(spi_handle_t *hperh, uint8_t *buf, uint16_t size)
{
	assert_param(IS_SPI(hperh->perh));

	if(hperh->state != SPI_STATE_READY)
		return BUSY;
	if(buf == NULL || size == 0)
		return ERROR;

	__LOCK(hperh);
	hperh->state    = SPI_STATE_BUSY_TX;
	hperh->err_code = SPI_ERROR_NONE;

	hperh->tx_buf   = buf;
	hperh->tx_size  = size;
	hperh->tx_count = size;
	hperh->rx_buf   = NULL;
	hperh->rx_size  = 0;
	hperh->rx_count = 0;

	if (hperh->init.crc_calc)
		SPI_CRC_RESET(hperh);

	if (hperh->init.dir == SPI_DIRECTION_1LINE_TX)
		SPI_1LINE_TX(hperh);

	if (hperh->init.dir == SPI_DIRECTION_2LINES) {
		spi_interrupt_config(hperh, SPI_IT_TXE, ENABLE);
	}
	else {
//		spi_interrupt_config(hperh, SPI_IT_TXE, ENABLE);
//		spi_interrupt_config(hperh, SPI_IT_MODF, ENABLE);
	}

	__UNLOCK(hperh);

	if(hperh->perh->CR1.SPE != ENABLE)
		SPI_ENABLE(hperh);

	spi_interrupt_config(hperh, SPI_IT_TXE, ENABLE);

	return OK;
}

/**
  * @brief  Receives an amount of data in non blocking mode
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @param  buf: Pointer to data received buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t spi_recv_by_it(spi_handle_t *hperh, uint8_t *buf, uint16_t size)
{
	assert_param(IS_SPI(hperh->perh));

	if (hperh->state != SPI_STATE_READY)
		return BUSY;
	if (buf == NULL || size == 0)
		return ERROR;

	__LOCK(hperh);
	hperh->state    = SPI_STATE_BUSY_RX;
	hperh->err_code = SPI_ERROR_NONE;

	hperh->rx_buf   = buf;
	hperh->rx_size  = size;
	hperh->rx_count = size;
	hperh->tx_buf   = NULL;
	hperh->tx_size  = 0;
	hperh->tx_count = 0;

	if (hperh->init.dir == SPI_DIRECTION_1LINE_RX)
		SPI_1LINE_RX(hperh);

	if ((hperh->init.dir == SPI_DIRECTION_2LINES) && (hperh->init.mode == SPI_MODE_MASTER)) {
		__UNLOCK(hperh);
		return spi_send_recv_by_it(hperh, buf, buf, size);
	}

	if (hperh->init.crc_calc == ENABLE) {
		SPI_CRC_RESET(hperh);
		spi_interrupt_config(hperh, SPI_IT_CRCERR, ENABLE);
	}

//	spi_interrupt_config(hperh, SPI_IT_RXTH, ENABLE);
//	spi_interrupt_config(hperh, SPI_IT_MODF, ENABLE);
//	spi_interrupt_config(hperh, SPI_IT_OVR, ENABLE);

	__UNLOCK(hperh);

	if (hperh->perh->CR1.SPE != ENABLE)
		SPI_ENABLE(hperh);

	spi_interrupt_config(hperh, SPI_IT_RXTH, ENABLE);

	return OK;
}

/**
  * @brief  Transmit and Receives an amount of data in non blocking mode
  * @param  hperh: Pointer to a spi_handle_t structure that contains
  *         the configuration information for the specified SPI module.
  * @param  tx_buf: Pointer to data transmitted buffer
  * @param  rx_buf: Pointer to data received buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t spi_send_recv_by_it(spi_handle_t *hperh, uint8_t *tx_buf,uint8_t *rx_buf, uint16_t size)
{
	assert_param(IS_SPI(hperh->perh));

	if (hperh->state != SPI_STATE_READY)
		return BUSY;
	if (tx_buf == NULL || rx_buf == NULL || size == 0)
		return ERROR;

	__LOCK(hperh);
	hperh->state    = SPI_STATE_BUSY_TX_RX;
	hperh->err_code = SPI_ERROR_NONE;

	hperh->tx_buf   = tx_buf;
	hperh->tx_size  = size;
	hperh->tx_count = size - 1;
	hperh->rx_buf   = rx_buf;
	hperh->rx_size  = size;
	hperh->rx_count = size;

	if (hperh->init.crc_calc) {
		SPI_CRC_RESET(hperh);
		spi_interrupt_config(hperh, SPI_IT_CRCERR, ENABLE);
	}

	spi_interrupt_config(hperh, SPI_IT_TXE, ENABLE);
 	spi_interrupt_config(hperh, SPI_IT_MODF, ENABLE);

	__UNLOCK(hperh);

	if (hperh->perh->CR1.SPE != ENABLE)
		SPI_ENABLE(hperh);

	if (hperh->init.data_size <= SPI_DATA_SIZE_8) {
		*((volatile uint8_t *)hperh->perh + 0x0c) = *hperh->tx_buf;
		++hperh->tx_buf;
	}
	else {
		hperh->perh->DR.DR = (*(uint16_t *)hperh->tx_buf);
		hperh->tx_buf +=2;
	}
	return OK;
}

#ifdef HAL_DMA
/**
  * @brief  Transmit an amount of data used dma channel
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @param  channel: DMA channel as SPI transmit
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t spi_send_by_dma(spi_handle_t *hperh, uint8_t *buf, uint16_t size, uint8_t channel)
{
	assert_param(IS_SPI(hperh->perh));

	if (hperh->state != SPI_STATE_READY)
		return BUSY;
	if (buf == NULL|| size == 0)
		return ERROR;

	__LOCK(hperh);
	hperh->state    = SPI_STATE_BUSY_TX;
	hperh->err_code = SPI_ERROR_NONE;

	hperh->tx_buf   = buf;
	hperh->tx_size  = size;
	hperh->tx_count = size;
	hperh->rx_buf   = NULL;
	hperh->rx_size  = 0;
	hperh->rx_count = 0;

	if (hperh->init.dir == SPI_DIRECTION_1LINE_TX)
		SPI_1LINE_TX(hperh);
	if (hperh->init.crc_calc)
		SPI_CRC_RESET(hperh);

	hperh->hdmatx.cplt_arg = (void *)hperh;
	hperh->hdmatx.cplt_cbk = spi_dma_send_cplt;
	hperh->hdmatx.err_arg  = (void *)hperh;
	hperh->hdmatx.err_cbk  = spi_dma_error;

	/* Configure SPI DMA transmit */
	dma_config_struct(&(hperh->hdmatx.config));

	if (hperh->init.data_size <= SPI_DATA_SIZE_8) {
		hperh->hdmatx.config.data_width = DMA_DATA_SIZE_BYTE;
		hperh->hdmatx.config.src_inc    = DMA_DATA_INC_BYTE;
	}
	else {
		hperh->hdmatx.config.data_width = DMA_DATA_SIZE_HALFWORD;
		hperh->hdmatx.config.src_inc    = DMA_DATA_INC_HALFWORD;
	}

	hperh->hdmatx.perh              = DMA0;
	hperh->hdmatx.config.src        = (void *)buf;
	hperh->hdmatx.config.dst        = (void *)&hperh->perh->DR.Word;
	hperh->hdmatx.config.size       = size;
	hperh->hdmatx.config.dst_inc    = DMA_DATA_INC_NONE;
	hperh->hdmatx.config.msel       = hperh->perh == SPI0 ? DMA_MSEL_SPI0 : (hperh->perh == SPI1 ? DMA_MSEL_SPI1 : DMA_MSEL_SPI2);
	hperh->hdmatx.config.msigsel    = DMA_MSIGSEL_SPI_TXEMPTY;
	hperh->hdmatx.config.channel    = channel;
	hperh->hdmatx.config.burst      = ENABLE;
	dma_config_basic(&(hperh->hdmatx));

	spi_dma_req_config(hperh, SPI_DMA_REQ_TX, ENABLE);
	__UNLOCK(hperh);

	if (hperh->perh->CR1.SPE != ENABLE)
		SPI_ENABLE(hperh);

	return OK;
}

/**
  * @brief  Receive an amount of data used dma channel
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @param  channel: DMA channel as SPI transmit
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t spi_recv_by_dma(spi_handle_t *hperh, uint8_t *buf, uint16_t size, uint8_t channel)
{
	assert_param(IS_SPI(hperh->perh));

	if (hperh->state != SPI_STATE_READY)
		return BUSY;
	if (buf == NULL|| size == 0)
		return ERROR;

	__LOCK(hperh);
	hperh->state    = SPI_STATE_BUSY_RX;
	hperh->err_code = SPI_ERROR_NONE;

	hperh->rx_buf   = buf;
	hperh->rx_size  = size;
	hperh->rx_count = size;
	hperh->tx_buf   = NULL;
	hperh->tx_size  = 0;
	hperh->tx_count = 0;

	if (hperh->init.dir == SPI_DIRECTION_1LINE_RX)
		SPI_1LINE_RX(hperh);
	if ((hperh->init.dir == SPI_DIRECTION_2LINES) && (hperh->init.mode == SPI_MODE_MASTER)) {
		__UNLOCK(hperh);
		return ERROR;	/* Please use spi_send_recv_by_dma() */
	}
	if (hperh->init.crc_calc)
		SPI_CRC_RESET(hperh);

	hperh->hdmarx.cplt_arg = (void *)hperh;
	hperh->hdmarx.cplt_cbk = spi_dma_recv_cplt;
	hperh->hdmarx.err_arg  = (void *)hperh;
	hperh->hdmarx.err_cbk  = spi_dma_error;

	/* Configure DMA Receive */
	dma_config_struct(&(hperh->hdmarx.config));

	if (hperh->init.data_size <= SPI_DATA_SIZE_8) {
		hperh->hdmarx.config.data_width = DMA_DATA_SIZE_BYTE;
		hperh->hdmarx.config.dst_inc    = DMA_DATA_INC_BYTE;
	}
	else {
		hperh->hdmarx.config.data_width = DMA_DATA_SIZE_HALFWORD;
		hperh->hdmarx.config.dst_inc    = DMA_DATA_INC_HALFWORD;
	}

	hperh->hdmarx.perh              = DMA0;
	hperh->hdmarx.config.src        = (void *)&hperh->perh->DR.Word;
	hperh->hdmarx.config.dst        = (void *)buf;
	hperh->hdmarx.config.size       = size;
	hperh->hdmarx.config.src_inc    = DMA_DATA_INC_NONE;
	hperh->hdmarx.config.msel       = hperh->perh == SPI0 ? DMA_MSEL_SPI0 : (hperh->perh == SPI1 ? DMA_MSEL_SPI1 : DMA_MSEL_SPI2);
	hperh->hdmarx.config.msigsel    = DMA_MSIGSEL_SPI_RNR;
	hperh->hdmarx.config.channel    = channel;
	dma_config_basic(&(hperh->hdmarx));
	__UNLOCK(hperh);

	if (hperh->perh->CR1.SPE != ENABLE)
		SPI_ENABLE(hperh);

	spi_dma_req_config(hperh, SPI_DMA_REQ_RX, ENABLE);

	return OK;
}

/**
  * @brief  Transmit and Receive an amount of data used dma channel
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @param  tx_buf: Pointer to data buffer
  * @param  rx_buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @param  tx_channel: DMA channel as SPI transmit
  * @param  rx_channel: DMA channel as SPI receive
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t spi_send_recv_by_dma(spi_handle_t *hperh, uint8_t *tx_buf, uint8_t *rx_buf, uint16_t size, uint8_t tx_channel, uint8_t rx_channel)
{
	assert_param(IS_SPI(hperh->perh));

	if (hperh->state != SPI_STATE_READY && hperh->state != SPI_STATE_BUSY_RX)
		return BUSY;
	if (tx_buf == NULL || rx_buf == NULL || size == 0)
		return ERROR;

	__LOCK(hperh);
	hperh->state    = SPI_STATE_BUSY_RX;
	hperh->err_code = SPI_ERROR_NONE;

	hperh->tx_buf   = tx_buf;
	hperh->tx_size  = size;
	hperh->tx_count = size;
	hperh->rx_buf   = rx_buf;
	hperh->rx_size  = size;
	hperh->rx_count = size;

	hperh->hdmatx.cplt_arg = NULL;
	hperh->hdmatx.cplt_cbk = NULL;
	hperh->hdmatx.err_arg  = (void *)hperh;
	hperh->hdmatx.err_cbk  = spi_dma_error;
	hperh->hdmarx.cplt_arg = (void *)hperh;
	hperh->hdmarx.cplt_cbk = spi_dma_send_recv_cplt;
	hperh->hdmarx.err_arg  = (void *)hperh;
	hperh->hdmarx.err_cbk  = spi_dma_error;

	if (hperh->init.crc_calc)
		SPI_CRC_RESET(hperh);

	/* Configure SPI DMA transmit */
	dma_config_struct(&(hperh->hdmatx.config));
	dma_config_struct(&(hperh->hdmarx.config));

	if (hperh->init.data_size <= SPI_DATA_SIZE_8) {
		hperh->hdmatx.config.data_width = DMA_DATA_SIZE_BYTE;
		hperh->hdmarx.config.data_width = DMA_DATA_SIZE_BYTE;
		hperh->hdmatx.config.src_inc    = DMA_DATA_INC_BYTE;
		hperh->hdmarx.config.dst_inc    = DMA_DATA_INC_BYTE;
	}
	else {
		hperh->hdmatx.config.data_width = DMA_DATA_SIZE_HALFWORD;
		hperh->hdmarx.config.data_width = DMA_DATA_SIZE_HALFWORD;
		hperh->hdmatx.config.src_inc    = DMA_DATA_INC_HALFWORD;
		hperh->hdmarx.config.dst_inc    = DMA_DATA_INC_HALFWORD;
	}

	hperh->hdmatx.config.src        = (void *)tx_buf;
	hperh->hdmatx.config.dst        = (void *)&hperh->perh->DR.Word;
	hperh->hdmatx.config.size       = size;
	hperh->hdmatx.config.dst_inc    = DMA_DATA_INC_NONE;
	hperh->hdmatx.config.msel       = hperh->perh == SPI0 ? DMA_MSEL_SPI0 : (hperh->perh == SPI1 ? DMA_MSEL_SPI1 : DMA_MSEL_SPI2);
	hperh->hdmatx.config.msigsel    = DMA_MSIGSEL_SPI_TXEMPTY;
	hperh->hdmatx.config.channel    = tx_channel;
	hperh->hdmatx.config.burst      = ENABLE;
	dma_config_basic(&(hperh->hdmatx));

	/* Configure DMA Receive */
	hperh->hdmarx.config.src        = (void *)&hperh->perh->DR.Word;
	hperh->hdmarx.config.dst        = (void *)rx_buf;
	hperh->hdmarx.config.size       = size;
	hperh->hdmarx.config.src_inc    = DMA_DATA_INC_NONE;
	hperh->hdmarx.config.msel       = hperh->perh == SPI0 ? DMA_MSEL_SPI0 : (hperh->perh == SPI1 ? DMA_MSEL_SPI1 : DMA_MSEL_SPI2);
	hperh->hdmarx.config.msigsel    = DMA_MSIGSEL_SPI_RNR;
	hperh->hdmarx.config.channel    = rx_channel;
	hperh->hdmarx.config.burst      = ENABLE;
	dma_config_basic(&(hperh->hdmarx));

	spi_dma_req_config(hperh, SPI_DMA_REQ_TX, ENABLE);
	spi_dma_req_config(hperh, SPI_DMA_REQ_RX, ENABLE);
	__UNLOCK(hperh);

	if(hperh->perh->CR1.SPE != ENABLE)
		SPI_ENABLE(hperh);

	return OK;
}

/**
  * @brief  Pauses the DMA Transfer.
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @retval Status
  */
hal_status_t spi_dma_pause(spi_handle_t *hperh)
{
	assert_param(IS_SPI(hperh->perh));

	__LOCK(hperh);
	spi_dma_req_config(hperh, SPI_DMA_REQ_TX, DISABLE);
	spi_dma_req_config(hperh, SPI_DMA_REQ_RX, DISABLE);
	__UNLOCK(hperh);

	return OK;
}

/**
  * @brief  Resumes the DMA Transfer.
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @retval Status
  */
hal_status_t spi_dma_resume(spi_handle_t *hperh)
{
	assert_param(IS_SPI(hperh->perh));

	__LOCK(hperh);
	spi_dma_req_config(hperh, SPI_DMA_REQ_TX, ENABLE);
	spi_dma_req_config(hperh, SPI_DMA_REQ_RX, ENABLE);
	__UNLOCK(hperh);

	return OK;
}

/**
  * @brief  Stops the DMA Transfer.
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @retval Status
  */
hal_status_t spi_dma_stop(spi_handle_t *hperh)
{
	assert_param(IS_SPI(hperh->perh));

	__LOCK(hperh);
	spi_dma_req_config(hperh, SPI_DMA_REQ_TX, DISABLE);
	spi_dma_req_config(hperh, SPI_DMA_REQ_RX, DISABLE);
	__UNLOCK(hperh);

	hperh->state = SPI_STATE_READY;
	return OK;
}
#endif
/**
  * @}
  */

/** @defgroup SPI_Public_Functions_Group3 Control functions
  * @brief SPI Control functions
  *
  * @verbatim
   ===============================================================================
                      ##### Peripheral Control functions #####
   ===============================================================================
    [..]
    This subsection provides a set of functions allowing to control the SPI.
     (+) Handle interrupt about SPI module. The spi_irq_handle() function must
         be invoked by SPI-IRQ function.
     (+) Configure the interrupt DISABLE/ENABLE.
     (+) Configure the DMA request.
     (+) Get interrupt source status.
     (+) Get interrupt flag status.
     (+) Clear interrupt flag

    @endverbatim
  * @{
  */

/**
  * @brief  This function handles SPI interrupt request.
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @retval None
  */
void spi_irq_handle(spi_handle_t *hperh)
{
	if ((spi_get_it_status(hperh, SPI_IT_RXTH) != RESET) && (spi_get_it_flag_status(hperh, SPI_IF_RXTH) != RESET)) {
		if (hperh->state == SPI_STATE_BUSY_RX)
			__spi_recv_by_it(hperh);
		else
			__spi_send_recv_by_it(hperh);

		return;
	}
	if ((spi_get_it_status(hperh, SPI_IT_TXE) != RESET) && (spi_get_it_flag_status(hperh, SPI_IF_TXE) != RESET)) {
		spi_clear_it_flag_status(hperh, SPI_IF_TXE);
		if (hperh->state == SPI_STATE_BUSY_TX)
			__spi_send_by_it(hperh);
		else
			__spi_send_recv_by_it(hperh);

		return;
	}


	if ((spi_get_it_status(hperh, SPI_IT_CRCERR) != RESET) && (spi_get_it_flag_status(hperh, SPI_IF_CRCERR) != RESET)) {
		hperh->err_code |= SPI_ERROR_CRC;
		spi_clear_it_flag_status(hperh, SPI_IF_CRCERR);
	}
	if ((spi_get_it_status(hperh, SPI_IT_MODF) != RESET) && (spi_get_it_flag_status(hperh, SPI_IF_MODF) != RESET)) {
		hperh->err_code |= SPI_ERROR_MODF;
		spi_clear_it_flag_status(hperh, SPI_IF_MODF);
	}

	return;
}

/**
  * @brief  Enables or disables the specified SPI interrupts.
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @param  it: Specifies the SPI interrupt sources to be enabled or disabled.
  *         This parameter can be one of the @ref spi_it_t.
  * @param  state: New status
  *           - ENABLE
  *           - DISABLE
  * @retval None
  */
void spi_interrupt_config(spi_handle_t *hperh, spi_it_t it, type_func_t state)
{
	assert_param(IS_SPI(hperh->perh));
	assert_param(IS_SPI_IT(it));
	assert_param(IS_FUNC_STATE(state));

	if (state == ENABLE)
		hperh->perh->IER.Word |= (uint32_t)it;
	else
		hperh->perh->IDR.Word |= (uint32_t)it;

	return;
}

/**
  * @brief  Enables or disables the dma request.
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @param  req: Specifies the SPI dma request sources to be enabled or disabled.
  *         This parameter can be one of the @ref spi_dma_req_t.
  * @param  state: New status
  *           - ENABLE
  *           - DISABLE
  * @retval None
  */
void spi_dma_req_config(spi_handle_t *hperh, spi_dma_req_t req, type_func_t state)
{
	assert_param(IS_SPI(hperh->perh));
	assert_param(IS_SPI_DMA_REQ(req));
	assert_param(IS_FUNC_STATE(state));

	if (state == ENABLE) {
		if (req == SPI_DMA_REQ_TX)
			hperh->perh->CR2.TXDMAEN = 1;
		else
			hperh->perh->CR2.RXDMAEN = 1;
	}
	else {
		if (req == SPI_DMA_REQ_TX)
			hperh->perh->CR2.TXDMAEN = 0;
		else
			hperh->perh->CR2.RXDMAEN = 0;
	}

	return;
}

/** @brief  Check whether the specified SPI flag is set or not.
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @param  flag: specifies the flag to check.
  *         This parameter can be one of the @ref spi_flag_t.
  * @retval Status
  *           - SET
  *           - RESET
  */
flag_status_t spi_get_flag_status(spi_handle_t *hperh, spi_flag_t flag)
{
	assert_param(IS_SPI(hperh->perh));
	assert_param(IS_SPI_FLAG(flag));

	if (hperh->perh->SR.Word & flag)
		return SET;

	return RESET;
}

/**
  * @brief  Checks whether the specified SPI interrupt has occurred or not.
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @param  it: Specifies the SPI interrupt source to check.
  *         This parameter can be one of the @ref spi_it_t.
  * @retval Status
  *           - SET
  *           - RESET
  */
it_status_t spi_get_it_status(spi_handle_t *hperh, spi_it_t it)
{
	assert_param(IS_SPI(hperh->perh));
	assert_param(IS_SPI_IT(it));

	if (hperh->perh->IVS.Word & it)
		return SET;

	return RESET;
}

/** @brief  Check whether the specified SPI interrupt flag is set or not.
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @param  flag: specifies the flag to check.
  *         This parameter can be one of the @ref spi_it_flag_t.
  * @retval Status
  *           - SET
  *           - RESET
  */
flag_status_t spi_get_it_flag_status(spi_handle_t *hperh, spi_it_flag_t flag)
{
	assert_param(IS_SPI(hperh->perh));
	assert_param(IS_SPI_IF(flag));

	if (hperh->perh->RIF.Word & flag)
		return SET;

	return RESET;
}

/** @brief  Clear the specified SPI interrupt flags.
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @param  flag: specifies the flag to check.
  *         This parameter can be one of the @ref spi_it_flag_t.
  * @retval None
  */
void spi_clear_it_flag_status(spi_handle_t *hperh, spi_it_flag_t flag)
{
	assert_param(IS_SPI(hperh->perh));
	assert_param(IS_SPI_IF(flag));

	hperh->perh->ICR.Word |= flag;
	return;
}

/**
  * @brief  This function handles SPI Communication Timeout.
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @param  flag: specifies the SPI flag to check.
  * @param  status: The new Flag status (SET or RESET).
  * @param  timeout: Timeout duration
  * @retval Status, see @ref hal_status_t.
  */
static hal_status_t spi_wait_flag(spi_handle_t *hperh, spi_flag_t flag, flag_status_t status, uint32_t timeout)
{
	uint32_t tick = __get_tick();

	assert_param(timeout > 0);

	while ((spi_get_flag_status(hperh, flag)) != status) {
		if (((__get_tick()) - tick) > timeout) {
			spi_interrupt_config(hperh, SPI_IT_TXE, DISABLE);
			spi_interrupt_config(hperh, SPI_IT_RXF, DISABLE);
			spi_interrupt_config(hperh, SPI_IT_CRCERR, DISABLE);
			spi_interrupt_config(hperh, SPI_IT_MODF, DISABLE);
			return TIMEOUT;
		}
	}

	return OK;
}

static hal_status_t spi_wait_bsy_flag(spi_handle_t *hperh,flag_status_t status, uint32_t timeout)
{
	uint32_t tick = __get_tick();

	assert_param(timeout > 0);

	while (hperh->perh->SR.BSY != status) {
		if (((__get_tick()) - tick) > timeout) {
			spi_interrupt_config(hperh, SPI_IT_TXE, DISABLE);
			spi_interrupt_config(hperh, SPI_IT_RXF, DISABLE);
			spi_interrupt_config(hperh, SPI_IT_CRCERR, DISABLE);
			spi_interrupt_config(hperh, SPI_IT_MODF, DISABLE);
			return TIMEOUT;
		}
	}

	return OK;
}
/**
  * @}
  */

/** @defgroup SPI_Public_Functions_Group4 Peripheral State and Errors functions
  *  @brief   SPI State and Errors functions
  *
  * @verbatim
 ===============================================================================
                      ##### Peripheral State and Errors functions #####
 ===============================================================================
    [..]
    This subsection provides a set of functions allowing to control the SPI.
     (+) spi_get_state() API can check in run-time the state of the SPI peripheral
     (+) spi_get_error() check in run-time Errors occurring during communication

    @endverbatim
  * @{
  */

/**
  * @brief  Returns the SPI state.
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @retval HAL state
  */
spi_state_t spi_get_state(spi_handle_t *hperh)
{
	assert_param(IS_SPI(hperh->perh));
	return hperh->state;
}

/**
  * @brief  Return the SPI error code
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @retval SPI Error Code
  */
uint32_t spi_get_error(spi_handle_t *hperh)
{
	assert_param(IS_SPI(hperh->perh));
	return hperh->err_code;
}
/**
  * @}
  */

/**
  * @}
  */

/** @defgroup SPI_Private_Functions SPI Private Functions
  * @brief   SPI Private functions
  * @{
  */

/**
  * @brief  handle program when an tx empty interrupt flag arrived in non block mode
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @retval Status, see @ref hal_status_t.
  */
static void __spi_send_by_it(spi_handle_t *hperh)
{
	if (hperh->tx_count == 0) {
		spi_interrupt_config(hperh, SPI_IT_TXE, DISABLE);
		hperh->state = SPI_STATE_READY;


		if ((spi_wait_bsy_flag(hperh, RESET, 1000)) != OK) {
			if (hperh->err_cbk)
				hperh->err_cbk(hperh);

			return;
		}

		if (hperh->tx_cplt_cbk)
			hperh->tx_cplt_cbk(hperh);

		return;
	}

	if (hperh->init.data_size <= SPI_DATA_SIZE_8) {
		*((volatile uint8_t *)hperh->perh + 0x0c) = *hperh->tx_buf;
		++hperh->tx_buf;
	}
	else {
		hperh->perh->DR.DR = *(uint16_t *)hperh->tx_buf;
		hperh->tx_buf += 2;
	}
	--hperh->tx_count;

	if (hperh->tx_count == 0) {
		if(hperh->init.crc_calc)
			SPI_CRCNEXT_ENABLE(hperh);
	}

	return;
}

/**
  * @brief  handle program when an rx no empty interrupt flag arrived in non block mode
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @retval Status, see @ref hal_status_t.
  */
static void __spi_recv_by_it(spi_handle_t *hperh)
{
	if (hperh->init.data_size <= SPI_DATA_SIZE_8) {
		*hperh->rx_buf = hperh->perh->DR.DR;
		++hperh->rx_buf;
	}
	else {
		*(uint16_t *)hperh->rx_buf = hperh->perh->DR.DR;
		hperh->rx_buf += 2;
	}

	spi_clear_it_flag_status(hperh, SPI_IF_RXTH);
	--hperh->rx_count;

	if (hperh->rx_count == 0) {
		spi_interrupt_config(hperh, SPI_IT_RXTH, DISABLE);
		hperh->state = SPI_STATE_READY;

		if ((hperh->init.crc_calc) && (spi_get_it_flag_status(hperh, SPI_IF_CRCERR) != RESET)) {
			hperh->err_code |= SPI_ERROR_CRC;
			spi_clear_it_flag_status(hperh, SPI_IF_CRCERR);

			if (hperh->err_cbk)
				hperh->err_cbk(hperh);

			return;
		}

		if (hperh->rx_cplt_cbk)
			hperh->rx_cplt_cbk(hperh);
	}

	return;
}

/**
  * @brief  handle program when an rx no empty interrupt flag arrived in non block mode(2 lines)
  * @param  hperh: Pointer to a spi_handle_t structure.
  * @retval Status, see @ref hal_status_t.
  */
static void __spi_send_recv_by_it(spi_handle_t *hperh)
{
	if (hperh->tx_count == hperh->tx_size - 1) {
		if ((spi_get_it_flag_status(hperh, SPI_IF_RXF)) == SET) {
			if (hperh->init.data_size <= SPI_DATA_SIZE_8) {
				*hperh->rx_buf = hperh->perh->DR.DR;
				++hperh->rx_buf;
			}
			else {
				*(uint16_t *)hperh->rx_buf = hperh->perh->DR.DR;
				hperh->rx_buf += 2;
			}

			if ((--hperh->rx_count == 1) && (hperh->init.crc_calc))
				SPI_CRCNEXT_ENABLE(hperh);
		}
	}
	if (hperh->tx_count != 0) {
		if ((spi_get_it_flag_status(hperh, SPI_IF_TXE)) == SET) {
			if (hperh->init.data_size <= SPI_DATA_SIZE_8) {
				*((volatile uint8_t *)hperh->perh + 0x0c) = *hperh->tx_buf;
				++hperh->tx_buf;
			}
			else {
				hperh->perh->DR.DR = *(uint16_t *)hperh->tx_buf;
				hperh->tx_buf += 2;
			}

			if (--hperh->tx_count == 0)
				spi_interrupt_config(hperh, SPI_IT_TXE, DISABLE);
		}
	}

	if (hperh->rx_count != 0) {
		if ((spi_get_it_flag_status(hperh, SPI_IF_RXF)) == SET) {
			if (hperh->init.data_size <= SPI_DATA_SIZE_8) {
				*hperh->rx_buf = hperh->perh->DR.DR;
				++hperh->rx_buf;
			}
			else {
				*(uint16_t *)hperh->rx_buf = hperh->perh->DR.DR;
				hperh->rx_buf += 2;
			}

			if ((--hperh->rx_count == 1) && (hperh->init.crc_calc))
				SPI_CRCNEXT_ENABLE(hperh);
		}
	}

	if (hperh->rx_count == 0) {
		spi_interrupt_config(hperh, SPI_IT_TXE, DISABLE);
		spi_interrupt_config(hperh, SPI_IT_RXF, DISABLE);
		spi_interrupt_config(hperh, SPI_IT_CRCERR, DISABLE);
		spi_interrupt_config(hperh, SPI_IT_MODF, DISABLE);
		hperh->state = SPI_STATE_READY;

		if ((hperh->init.crc_calc) && (spi_get_it_flag_status(hperh, SPI_IF_CRCERR) != RESET)) {
			hperh->err_code |= SPI_ERROR_CRC;
			spi_clear_it_flag_status(hperh, SPI_IF_CRCERR);

			if (hperh->err_cbk)
				hperh->err_cbk(hperh);

			return;
		}

		if (hperh->rx_cplt_cbk)
			hperh->rx_cplt_cbk(hperh);
	}

	return;
}


#ifdef HAL_DMA
/**
  * @brief  DMA SPI transmit process complete callback.
  * @param  arg: Pointer to a spi_handle_t structure.
  * @retval None
  */
static void spi_dma_send_cplt(void *arg)
{
	spi_handle_t *hperh = (spi_handle_t *)arg;

	hperh->tx_count = 0;
	spi_dma_req_config(hperh, SPI_DMA_REQ_TX, DISABLE);
	hperh->state = SPI_STATE_READY;


	if ((spi_wait_bsy_flag(hperh, RESET, 1000)) != OK)
		hperh->err_code |= SPI_ERROR_FLAG;

	if (hperh->err_code == SPI_ERROR_NONE) {
		if (hperh->tx_cplt_cbk)
			hperh->tx_cplt_cbk(hperh);
	}
	else {
		if (hperh->err_cbk)
			hperh->err_cbk(hperh);
	}

	return;
}

/**
  * @brief  DMA SPI receive process complete callback.
  * @param  arg: Pointer to a spi_handle_t structure.
  * @retval None
  */
static void spi_dma_recv_cplt(void *arg)
{
	uint32_t tmp;
	spi_handle_t *hperh = (spi_handle_t *)arg;

	hperh->rx_count = 0;
	spi_dma_req_config(hperh, SPI_DMA_REQ_TX, DISABLE);
	spi_dma_req_config(hperh, SPI_DMA_REQ_RX, DISABLE);
	hperh->state = SPI_STATE_READY;

	if (hperh->init.crc_calc) {
		if ((spi_wait_flag(hperh, SPI_FLAG_RXF, SET, 1000)) != OK)
			hperh->err_code |= SPI_ERROR_FLAG;

		tmp = hperh->perh->DR.Word;
		UNUSED(tmp);

	}

	if ((hperh->init.mode == SPI_MODE_MASTER) && (hperh->init.dir != SPI_DIRECTION_2LINES))
		SPI_DISABLE(hperh);

	if (hperh->err_code == SPI_ERROR_NONE) {
		if (hperh->rx_cplt_cbk)
			hperh->rx_cplt_cbk(hperh);
	}
	else {
		if (hperh->err_cbk)
			hperh->err_cbk(hperh);
	}

	return;
}

/**
  * @brief  DMA SPI transmit and receive process complete callback.
  * @param  arg: Pointer to a SPI_handle_t structure.
  * @retval None
  */
static void spi_dma_send_recv_cplt(void *arg)
{
	uint32_t tmp;
	spi_handle_t *hperh = (spi_handle_t *)arg;

	if (hperh->init.crc_calc) {
		if ((spi_wait_flag(hperh, SPI_FLAG_RXF, SET, 1000)) != OK)
			hperh->err_code |= SPI_ERROR_FLAG;

		tmp = hperh->perh->DR.Word;
		UNUSED(tmp);

	}

	if ((spi_wait_bsy_flag(hperh, RESET, 1000)) != OK)
		hperh->err_code |= SPI_ERROR_FLAG;

	spi_dma_req_config(hperh, SPI_DMA_REQ_TX, DISABLE);
	spi_dma_req_config(hperh, SPI_DMA_REQ_RX, DISABLE);
	hperh->tx_count = 0;
	hperh->rx_count = 0;
	hperh->state    = SPI_STATE_READY;

	if (hperh->err_code == SPI_ERROR_NONE) {
		if (hperh->tx_rx_cplt_cbk)
			hperh->tx_rx_cplt_cbk(hperh);
	}
	else {
		if (hperh->err_cbk)
			hperh->err_cbk(hperh);
	}

	return;
}

/**
  * @brief  DMA SPI communication error callback.
  * @param  arg: Pointer to a spi_handle_t structure that contains
  *         the configuration information for the specified SPI module.
  * @retval None
  */
static void spi_dma_error(void *arg)
{
	spi_handle_t *hperh = (spi_handle_t *)arg;

	spi_dma_req_config(hperh, SPI_DMA_REQ_TX, DISABLE);
	spi_dma_req_config(hperh, SPI_DMA_REQ_RX, DISABLE);
	SET_BIT(hperh->err_code, SPI_ERROR_DMA);

 	hperh->tx_count = 0;
 	hperh->rx_count = 0;
	hperh->state    = SPI_STATE_READY;

	if (hperh->err_cbk)
		hperh->err_cbk(hperh);

	return;
}
#endif /* HAL_DMA */
/**
  * @}
  */
#endif /* HAL_SPI */
/**
  * @}
  */

/**
  * @}
  */
