/**
  *********************************************************************************
  *
  * @file    hal_uart.h
  * @brief   Header file of UART module library.
  *
  * @version V1.0
  * @date    21 Nov 2017
  * @author  AE Team
  * @note
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  */

#ifndef __HAL_UART_H__
#define __HAL_UART_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "utils.h"
#include "hal_dma.h"


/** @addtogroup ES32FXXX_HAL
  * @{
  */

/** @addtogroup UART
  * @{
  */

/**
  * @defgroup UART_Public_Macros UART Public Macros
  * @{
  */
#define UART_RX_ENABLE(hperh)		((hperh)->perh->LCR.RXEN = 1)
#define UART_RX_DISABLE(hperh)		((hperh)->perh->LCR.RXEN = 0)
#define UART_TX_ENABLE(hperh)		((hperh)->perh->LCR.TXEN = 1)
#define UART_TX_DISABLE(hperh)		((hperh)->perh->LCR.TXEN = 0)
#define UART_RX_TIMEOUT_ENABLE(hperh)	((hperh)->perh->RTOR.RTOEN = 1)
#define UART_RX_TIMEOUT_DISABLE(hperh)	((hperh)->perh->RTOR.RTOEN = 0)
#define UART_MSB_FIRST_ENABLE(hperh)	((hperh)->perh->LCR.MSB = 1)
#define UART_MSB_FIRST_DISABLE(hperh)	((hperh)->perh->LCR.MSB = 0)
#define UART_DATA_INV_ENABLE(hperh)	((hperh)->perh->LCR.DATAINV = 1)
#define UART_DATA_INV_DISABLE(hperh)	((hperh)->perh->LCR.DATAINV = 0)
#define UART_RX_INV_ENABLE(hperh)	((hperh)->perh->LCR.RXINV = 1)
#define UART_RX_INV_DISABLE(hperh)	((hperh)->perh->LCR.RXINV = 0)
#define UART_TX_INV_ENABLE(hperh)	((hperh)->perh->LCR.TXINV = 1)
#define UART_TX_INV_DISABLE(hperh)	((hperh)->perh->LCR.TXINV = 0)
#define UART_TX_RX_SWAP_ENABLE(hperh)	((hperh)->perh->LCR.SWAP = 1)
#define UART_TX_RX_SWAP_DISABLE(hperh)	((hperh)->perh->LCR.SWAP = 0)
#define UART_HDSEL_ENABLE(hperh)	((hperh)->perh->MCR.HDEN = 1)
#define UART_HDSEL_DISABLE(hperh)	((hperh)->perh->MCR.HDEN = 0)
#define UART_FIFO_TX_RESET(hperh)	((hperh)->perh->FCR.TFRST = 1)
#define UART_FIFO_RX_RESET(hperh)	((hperh)->perh->FCR.RFRST = 1)
#define UART_LPBMOD_ENABLE(hperh)	((hperh)->perh->MCR.LPBKEN = 1)
#define UART_LPBMOD_DISABLE(hperh)	((hperh)->perh->MCR.LPBKEN = 0)
#define UART_AUTOBR_ENABLE(hperh)	((hperh)->perh->MCR.ABREN = 1)
#define UART_AUTOBR_DISABLE(hperh)	((hperh)->perh->MCR.ABREN = 0)
#define UART_AUTOBR_REPT(hperh)		((hperh)->perh->MCR.ABRREPT = 1)
#define UART_GET_BRR_VALUE(hperh)	((hperh)->perh->BRR.BRR)
#define UART_SET_TIMEOUT_VALUE(x, y)	((x)->perh->RTOR.RTO = y)
/**
  * @}
  */

/** @defgroup UART_Public_Types UART Public Types
  * @{
  */
/**
  * @brief UART word length
  */
typedef enum {
	UART_WORD_LENGTH_8B = 0x0,	/**< 8-bits */
	UART_WORD_LENGTH_7B = 0x1,	/**< 7-bits */
	UART_WORD_LENGTH_6B = 0x2,	/**< 6-bits */
	UART_WORD_LENGTH_5B = 0x3,	/**< 5-bits */
} uart_word_length_t;

/**
  * @brief UART stop bits
  */
typedef enum {
	UART_STOP_BITS_1   = 0x0,		/**< 1-bits */
	UART_STOP_BITS_2   = 0x1,		/**< 2-bits */
	UART_STOP_BITS_0_5 = 0x0,		/**< 0.5-bits, using smartcard mode */
	UART_STOP_BITS_1_5 = 0x1,		/**< 1.5-bits, using smartcard mode */
} uart_stop_bits_t;

/**
  * @brief UART parity
  */
typedef enum {
	UART_PARITY_NONE = 0x0,		/**< Not parity */
	UART_PARITY_ODD  = 0x1,		/**< Odd parity */
	UART_PARITY_EVEN = 0x3,		/**< Even parity */
} uart_parity_t;

/**
  * @brief UART mode
  */
typedef enum {
	UART_MODE_UART  = 0x0,		/**< UART */
	UART_MODE_LIN   = 0x1,		/**< LIN */
	UART_MODE_IrDA  = 0x2,		/**< IrDA */
	UART_MODE_RS485 = 0x3,		/**< RS485 */
} uart_mode_t;

/**
  * @brief UART hardware flow control
  */
typedef enum {
	UART_HW_FLOW_CTL_DISABLE = 0x0,	/**< Auto-flow-control disable */
	UART_HW_FLOW_CTL_ENABLE  = 0x1,	/**< Auto-flow-control enable */
} uart_hw_flow_ctl_t;

/**
  * @brief HAL UART state
  */
typedef enum {
	UART_STATE_RESET      = 0x00,	/**< Peripheral is not initialized */
	UART_STATE_READY      = 0x01,	/**< Peripheral Initialized and ready for use */
	UART_STATE_BUSY       = 0x02,	/**< an internal process is ongoing */
	UART_STATE_BUSY_TX    = 0x12,	/**< Data Transmission process is ongoing */
	UART_STATE_BUSY_RX    = 0x22,	/**< Data Reception process is ongoing */
	UART_STATE_BUSY_TX_RX = 0x32,	/**< Data Transmission Reception process is ongoing */
	UART_STATE_TIMEOUT    = 0x03,	/**< Timeout state */
	UART_STATE_ERROR      = 0x04,	/**< Error */
} uart_state_t;

/**
  * @brief UART error codes
  */
typedef enum {
	UART_ERROR_NONE = ((uint32_t)0x00),	/**< No error */
	UART_ERROR_PE   = ((uint32_t)0x01),	/**< Parity error */
	UART_ERROR_NE   = ((uint32_t)0x02),	/**< Noise error */
	UART_ERROR_FE   = ((uint32_t)0x04),	/**< frame error */
	UART_ERROR_ORE  = ((uint32_t)0x08),	/**< Overrun error */
	UART_ERROR_DMA  = ((uint32_t)0x10),	/**< DMA transfer error */
} uart_error_t;

/**
  * @brief UART init structure definition
  */
typedef struct {
	uint32_t baud;			/**< Specifies the uart communication baud rate */
	uart_word_length_t word_length;	/**< Specifies the number of data bits transmitted or received in a frame */
	uart_stop_bits_t stop_bits;	/**< Specifies the number of stop bits transmitted */
	uart_parity_t parity;		/**< Specifies the parity mode */
	uart_mode_t mode;		/**< Specifies uart mode */
	uart_hw_flow_ctl_t fctl;	/**< Specifies wether the hardware flow control mode is enabled or disabled */
} uart_init_t;

/**
  * @brief  UART handle structure definition
  */
typedef struct uart_handle_s {
	UART_TypeDef *perh;	/**< UART registers base address */
	uart_init_t init;	/**< UART communication parameters */
	uint8_t *tx_buf;	/**< Pointer to UART Tx transfer Buffer */
	uint16_t tx_size;	/**< UART Tx Transfer size */
	uint16_t tx_count;	/**< UART Tx Transfer Counter */
	uint8_t *rx_buf;	/**< Pointer to UART Rx transfer Buffer */
	uint16_t rx_size;	/**< UART Rx Transfer size */
	uint16_t rx_count;	/**< UART Rx Transfer Counter */
#ifdef HAL_DMA
	dma_handle_t hdmatx;	/**< UART Tx DMA Handle parameters */
	dma_handle_t hdmarx;	/**< UART Rx DMA Handle parameters */
#endif
	lock_state_t lock;	/**< Locking object */
	uart_state_t state;	/**< UART communication state */
	uart_error_t err_code;	/**< UART Error code */

	void (*tx_cplt_cbk)(struct uart_handle_s *arg);	/**< Tx completed callback */
	void (*rx_cplt_cbk)(struct uart_handle_s *arg);	/**< Rx completed callback */
	void (*error_cbk)(struct uart_handle_s *arg);	/**< error callback */
} uart_handle_t;

/**
  * @brief UART RS485 configure structure definition
  */
typedef struct {
	type_func_t normal;	/**< Normal mode */
	type_func_t dir;	/**< Auto-direction mode */
	type_func_t invert;	/**< Address detection invert */
	uint8_t addr;		/**< Address for compare */
} uart_rs485_config_t;

/**
  * @brief LIN detection break length
  */
typedef enum {
	LIN_BREAK_LEN_10B = 0x0,	/**< 10-bit break */
	LIN_BREAK_LEN_11B = 0x1,	/**< 11-bit break */
} uart_lin_break_len_t;

/**
  * @brief UART TXFIFO size
  */
typedef enum {
	UART_TXFIFO_EMPTY = 0x0,	/**< Empty */
	UART_TXFIFO_2BYTE = 0x1,	/**< 2-Bytes */
	UART_TXFIFO_4BYTE = 0x2,	/**< 4-Bytes */
	UART_TXFIFO_8BYTE = 0x3,	/**< 8-Bytes */
} uart_txfifo_t;

/**
  * @brief UART RXFIFO size
  */
typedef enum {
	UART_RXFIFO_1BYTE  = 0x0,	/**< 1-Byte */
	UART_RXFIFO_4BYTE  = 0x1,	/**< 4-Bytes */
	UART_RXFIFO_8BYTE  = 0x2,	/**< 8-Bytes */
	UART_RXFIFO_14BYTE = 0x3,	/**< 14-Bytes */
} uart_rxfifo_t;

/**
  * @brief UART auto-baud mode
  */
typedef enum {
	UART_ABRMOD_1_TO_0 = 0x0,	/**< Detect bit0:1, bit1:0 */
	UART_ABRMOD_1      = 0x1,	/**< Detect bit0:1 */
	UART_ABRMOD_0_TO_1 = 0x2,	/**< Detect bit0:0, bit1:1 */
} uart_auto_baud_mode_t;

/**
  * @brief UART DMA Requests
  */
typedef enum {
	UART_DMA_REQ_TX = 0x0,	/**< TX dma */
	UART_DMA_REQ_RX = 0x1,	/**< RX dma */
} uart_dma_req_t;

/**
  * @brief UART status types
  */
typedef enum {
	UART_STATUS_PERR    = (1U << 0),	/**< Parity error */
	UART_STATUS_FERR    = (1U << 1),	/**< Framing error */
	UART_STATUS_BKERR   = (1U << 2),	/**< Break error */
	UART_STATUS_CTSSTA  = (1U << 3),	/**< Clear to send status */
	UART_STATUS_RSBUSY  = (1U << 8),	/**< Receive shif register busy */
	UART_STATUS_RFEMPTY = (1U << 10),	/**< Receive FIFO empty */
	UART_STATUS_RFFULL  = (1U << 11),	/**< Receive FIFO full */
	UART_STATUS_RFOERR  = (1U << 12),	/**< Reveive FIFO overrun error */
	UART_STATUS_RFUERR  = (1U << 13),	/**< Receive FIFO underrun error */
	UART_STATUS_TSBUSY  = (1U << 14),	/**< Transmit shit register busy */
	UART_STATUS_TFEMPTY = (1U << 16),	/**< Transmit FIFO empty */
	UART_STATUS_TFFULL  = (1U << 17),	/**< Transmit FIFO full */
	UART_STATUS_TFOERR  = (1U << 18),	/**< Transmit FIFO overrun error */
} uart_status_t;

/**
  * @brief UART interrupt types
  */
typedef enum {
	UART_IT_RXBERR  = (1U << 0),	/**< Receiver byte error */
	UART_IT_ABEND   = (1U << 1),	/**< Auto-Baud rate detection end */
	UART_IT_ABTO    = (1U << 2),	/**< Auto-Baud rate detection timeout */
	UART_IT_DCTS    = (1U << 3),	/**< Delta CTS status */
	UART_IT_RXTO    = (1U << 4),	/**< Receiver timeout */
	UART_IT_ADDRM   = (1U << 5),	/**< Addredd match */
	UART_IT_LINBK   = (1U << 6),	/**< Lin break detection */
	UART_IT_EOB     = (1U << 7),	/**< End of block */
	UART_IT_NOISE   = (1U << 8),	/**< Start bit noise detection */
	UART_IT_RFTH    = (1U << 9),	/**< Receive FIFO trigger threshold */
	UART_IT_RFFULL  = (1U << 11),	/**< Receive FIFO full */
	UART_IT_RFOERR  = (1U << 12),	/**< Receive FIFO overrun */
	UART_IT_RFUERR  = (1U << 13),	/**< Reveive FIFO underrun */
	UART_IT_TSEMPTY = (1U << 14),	/**< Transmit shift register empty */
	UART_IT_TFTH    = (1U << 15),	/**< Transmit FIFO trigger threshold */
	UART_IT_TFEMPTY = (1U << 16),	/**< Transmit FIFO empty */
	UART_IT_TFOVER  = (1U << 18),	/**< Transmit FIFO overrun */
} uart_it_t;

/**
  * @brief UART flags types
  */
typedef enum {
	UART_IF_RXBERR  = (1U << 0),	/**< Receiver byte error */
	UART_IF_ABEND   = (1U << 1),	/**< Auto-Baud rate detection end */
	UART_IF_ABTO    = (1U << 2),	/**< Auto-Baud rate detection timeout */
	UART_IF_DCTS    = (1U << 3),	/**< Delta CTS status */
	UART_IF_RXTO    = (1U << 4),	/**< Receiver timeout */
	UART_IF_ADDRM   = (1U << 5),	/**< Addredd match */
	UART_IF_LINBK   = (1U << 6),	/**< Lin break detection */
	UART_IF_EOB     = (1U << 7),	/**< End of block */
	UART_IF_NOISE   = (1U << 8),	/**< Start bit noise detection */
	UART_IF_RFTH    = (1U << 9),	/**< Receive FIFO trigger threshold */
	UART_IF_RFFULL  = (1U << 11),	/**< Receive FIFO full */
	UART_IF_RFOERR  = (1U << 12),	/**< Receive FIFO overrun */
	UART_IF_RFUERR  = (1U << 13),	/**< Reveive FIFO underrun */
	UART_IF_TSEMPTY = (1U << 14),	/**< Transmit shift register empty */
	UART_IF_TFTH    = (1U << 15),	/**< Transmit FIFO trigger threshold */
	UART_IF_TFEMPTY = (1U << 16),	/**< Transmit FIFO empty */
	UART_IF_TFOVER  = (1U << 18),	/**< Transmit FIFO overrun */
} uart_flag_t;
/**
  * @}
  */

/** @defgroup UART_Private_Macros   UART Private Macros
  * @{
  */
#define IS_UART_ALL(x)	(((x) == UART0) || \
                         ((x) == UART1) || \
                         ((x) == UART2) || \
                         ((x) == UART3))
#define IS_UART_WORD_LENGTH(x)	(((x) == UART_WORD_LENGTH_5B) || \
                                 ((x) == UART_WORD_LENGTH_6B) || \
                                 ((x) == UART_WORD_LENGTH_7B) || \
                                 ((x) == UART_WORD_LENGTH_8B))
#define IS_UART_STOPBITS(x)	(((x) == UART_STOP_BITS_1)   || \
                                 ((x) == UART_STOP_BITS_2)   ||\
                                 ((x) == UART_STOP_BITS_0_5) ||\
                                 ((x) == UART_STOP_BITS_1_5))
#define IS_UART_PARITY(x)	(((x) == UART_PARITY_NONE)  || \
                                 ((x) == UART_PARITY_ODD) || \
                                 ((x) == UART_PARITY_EVEN))
#define IS_UART_MODE(x)		(((x) == UART_MODE_UART) || \
                                 ((x) == UART_MODE_LIN)  || \
                                 ((x) == UART_MODE_IrDA) || \
                                 ((x) == UART_MODE_RS485))
#define IS_UART_HARDWARE_FLOW_CONTROL(x) \
                                (((x) == UART_HW_FLOW_CTL_DISABLE) || \
                                 ((x) == UART_HW_FLOW_CTL_ENABLE))
#define IS_UART_LIN_BREAK_LEN(x)	(((x) == LIN_BREAK_LEN_10B) || \
                                         ((x) == LIN_BREAK_LEN_11B))
#define IS_UART_TXFIFO_TYPE(x)	(((x) == UART_TXFIFO_EMPTY) || \
                                 ((x) == UART_TXFIFO_2BYTE) || \
                                 ((x) == UART_TXFIFO_4BYTE) || \
                                 ((x) == UART_TXFIFO_8BYTE))
#define IS_UART_RXFIFO_TYPE(x)	(((x) == UART_RXFIFO_1BYTE) || \
                                 ((x) == UART_RXFIFO_4BYTE) || \
                                 ((x) == UART_RXFIFO_8BYTE) || \
                                 ((x) == UART_RXFIFO_14BYTE))
#define IS_UART_AUTO_BAUD_MODE(x)	(((x) == UART_ABRMOD_1_TO_0) || \
                                         ((x) == UART_ABRMOD_1)      || \
                                         ((x) == UART_ABRMOD_0_TO_1))
#define IS_UART_DMA_REQ(x)	(((x) == UART_DMA_REQ_TX) || \
                                 ((x) == UART_DMA_REQ_RX))
#define IS_UART_STATUS(x)	(((x) == UART_STATUS_PERR)    || \
                                 ((x) == UART_STATUS_FERR)    || \
                                 ((x) == UART_STATUS_BKERR)   || \
                                 ((x) == UART_STATUS_CTSSTA)  || \
                                 ((x) == UART_STATUS_RSBUSY)  || \
                                 ((x) == UART_STATUS_RFEMPTY) || \
                                 ((x) == UART_STATUS_RFFULL)  || \
                                 ((x) == UART_STATUS_RFOERR)  || \
                                 ((x) == UART_STATUS_RFUERR)  || \
                                 ((x) == UART_STATUS_TSBUSY)  || \
                                 ((x) == UART_STATUS_TFEMPTY) || \
                                 ((x) == UART_STATUS_TFFULL)  || \
                                 ((x) == UART_STATUS_TFOERR))
#define IS_UART_IT(x)		(((x) == UART_IT_RXBERR)  || \
                                 ((x) == UART_IT_ABEND)   || \
                                 ((x) == UART_IT_ABTO)    || \
                                 ((x) == UART_IT_DCTS)    || \
                                 ((x) == UART_IT_RXTO)    || \
                                 ((x) == UART_IT_ADDRM)   || \
                                 ((x) == UART_IT_LINBK)   || \
                                 ((x) == UART_IT_EOB)     || \
                                 ((x) == UART_IT_NOISE)   || \
                                 ((x) == UART_IT_RFTH)    || \
                                 ((x) == UART_IT_RFFULL)  || \
                                 ((x) == UART_IT_RFOERR)  || \
                                 ((x) == UART_IT_RFUERR)  || \
                                 ((x) == UART_IT_TSEMPTY) || \
                                 ((x) == UART_IT_TFTH)    || \
                                 ((x) == UART_IT_TFEMPTY) || \
                                 ((x) == UART_IT_TFOVER))
#define IS_UART_IF(x)		(((x) == UART_IF_RXBERR)  || \
                                 ((x) == UART_IF_ABEND)   || \
                                 ((x) == UART_IF_ABTO)    || \
                                 ((x) == UART_IF_DCTS)    || \
                                 ((x) == UART_IF_RXTO)    || \
                                 ((x) == UART_IF_ADDRM)   || \
                                 ((x) == UART_IF_LINBK)   || \
                                 ((x) == UART_IF_EOB)     || \
                                 ((x) == UART_IF_NOISE)   || \
                                 ((x) == UART_IF_RFTH)    || \
                                 ((x) == UART_IF_RFFULL)  || \
                                 ((x) == UART_IF_RFOERR)  || \
                                 ((x) == UART_IF_RFUERR)  || \
                                 ((x) == UART_IF_TSEMPTY) || \
                                 ((x) == UART_IF_TFTH)    || \
                                 ((x) == UART_IF_TFEMPTY) || \
                                 ((x) == UART_IF_TFOVER))

#define IS_UART_BAUDRATE(x)	(((x) > 0) && ((x) < 0x44AA21))
#define IS_UART_DATA(x)		((x) <= 0x1FF)

#define UART_STATE_TX_MASK	(1U << 4)
#define UART_STATE_RX_MASK	(1U << 5)
/**
  * @}
  */

/** @addtogroup UART_Public_Functions
  * @{
  */

/** @addtogroup UART_Public_Functions_Group1
  * @{
  */
/* Initialization functions */
void uart_init(uart_handle_t *hperh);
void uart_reset(uart_handle_t *hperh);
void uart_rs485_config(uart_handle_t *hperh, uart_rs485_config_t *config);
/**
  * @}
  */

/** @addtogroup UART_Public_Functions_Group2
  * @{
  */
/* IO operation functions */
hal_status_t uart_send(uart_handle_t *hperh, uint8_t *buf, uint16_t size, uint32_t timeout);
hal_status_t uart_recv(uart_handle_t *hperh, uint8_t *buf, uint16_t size, uint32_t timeout);
hal_status_t uart_send_by_it(uart_handle_t *hperh, uint8_t *buf, uint16_t size);
hal_status_t uart_recv_by_it(uart_handle_t *hperh, uint8_t *buf, uint16_t size);
#ifdef HAL_DMA
hal_status_t uart_send_by_dma(uart_handle_t *hperh, uint8_t *buf, uint16_t size, uint8_t channel);
hal_status_t uart_recv_by_dma(uart_handle_t *hperh, uint8_t *buf, uint16_t size, uint8_t channel);
hal_status_t uart_dma_pause(uart_handle_t *hperh);
hal_status_t uart_dma_resume(uart_handle_t *hperh);
hal_status_t uart_dma_stop(uart_handle_t *hperh);
#endif
void uart_irq_handle(uart_handle_t *hperh);
/**
  * @}
  */

/** @addtogroup UART_Public_Functions_Group3
  * @{
  */
/* Peripheral Control functions */
void uart_interrupt_config(uart_handle_t *hperh, uart_it_t it, type_func_t state);
void uart_dma_req_config(uart_handle_t *hperh, uart_dma_req_t req, type_func_t state);
void uart_tx_fifo_config(uart_handle_t *hperh, uart_rxfifo_t config, uint8_t level);
void uart_rx_fifo_config(uart_handle_t *hperh, uart_rxfifo_t config, uint8_t level);
void uart_lin_send_break(uart_handle_t *hperh);
void uart_lin_detect_break_len_config(uart_handle_t *hperh, uart_lin_break_len_t len);
void uart_auto_baud_config(uart_handle_t *hperh, uart_auto_baud_mode_t mode);
it_status_t uart_get_it_status(uart_handle_t *hperh, uart_it_t it);
flag_status_t uart_get_status(uart_handle_t *hperh, uart_status_t status);
flag_status_t uart_get_flag_status(uart_handle_t *hperh, uart_flag_t flag);
flag_status_t uart_get_mask_flag_status(uart_handle_t *hperh, uart_flag_t flag);
void uart_clear_flag_status(uart_handle_t *hperh, uart_flag_t flag);
/**
  * @}
  */

/** @addtogroup UART_Public_Functions_Group4
  * @{
  */
/* Peripheral State and Errors functions */
uart_state_t uart_get_state(uart_handle_t *hperh);
uint32_t uart_get_error(uart_handle_t *hperh);
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif

#endif /* __HAL_UART_H__ */
