/**
 *********************************************************************************
 *
 * @file    hal_wdt.c
 * @brief   WDT module driver.
 *
 * @version V1.0
 * @date    18 Dec 2017
 * @author  AE Team
 * @note
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 *********************************************************************************
 */
#include "hal_conf.h"
#include "hal_wdt.h"


/** @addtogroup ES32FXXX_HAL
 * @{
 */

/** @defgroup WDT WDT
 * @brief WDT module driver
 * @{
 */
#ifdef HAL_WDT
/**
 * @defgroup WDT_Private_Macros WDT Private Macros
 * @{
 */
#define WWDT_UNLOCK() {WWDT->LOCK.Word = 0x1ACCE551;}
#define WWDT_LOCK()   {WWDT->LOCK.Word = 0xFFFFFFFF;}
#define IWDT_UNLOCK() {IWDT->LOCK.Word = 0x1ACCE551;}
#define IWDT_LOCK()   {IWDT->LOCK.Word = 0xFFFFFFFF;}
/**
 * @}
 */

/** @defgroup WWDT_Public_Functions WWDT Public Functions
 * @brief Init and configure WWDT function
 * @{
 */
/**
 * @brief Initializes the WWDT according to the specified parameters.
 * @param load: Specifies the free-running downcounter value.
 * @param win: specifics the no dog windows,
 *        the parameter can be one of the following values:
 *        @arg @ref WWDT_WIN_25 No dog window size: 25%
 *        @arg @ref WWDT_WIN_50 No dog window size: 50%
 *        @arg @ref WWDT_WIN_75 No dog window size: 75%
 *        @arg @ref WWDT_WIN_00 No dog window size: 0%
 * @param interrupt: Enable or disable interrupt.
 * @retval None
 */
void wwdt_init(uint32_t load, wwdt_win_t win, type_func_t interrupt)
{
	assert_param(IS_WWDT_WIN_TYPE(win));
	assert_param(IS_FUNC_STATE(interrupt));

	WWDT_UNLOCK();
	WWDT->LOAD.Word   = load;
	WWDT->CON.WWDTWIN = win;
	WWDT->CON.CLKS    = 0;
	WWDT->CON.RSTEN   = ENABLE;
	WWDT->CON.IE      = interrupt;
	WWDT_LOCK();

	return;
}

/**
 * @brief Start the WWDT
 * @retval None
 */
void wwdt_start(void)
{
	WWDT_UNLOCK();
	WWDT->CON.EN = 1;
	WWDT_LOCK();

	return;
}

/**
 * @brief Get the free-running downcounter value
 * @retval Value
 */
uint32_t wwdt_get_value(void)
{
	return WWDT->VALUE.Word;
}

/**
 * @brief Get interrupt state
 * @retval Value
 */
it_status_t wwdt_get_flag_status(void)
{
	return (it_status_t)(WWDT->RIS.WWDTIF);
}

/**
 * @brief Refreshes the WWDT
 * @retval None
 */
void wwdt_feed_dog(void)
{
	WWDT_UNLOCK();
	WWDT->INTCLR.INTCLR = 0x1;
	WWDT_LOCK();

	return;
}
/**
 * @}
 */

/** @defgroup IWDT_Public_Functions IWDT Public Functions
 * @brief Init and configure IWDT function
 * @{
 */
/**
 * @brief Initializes the IWDG according to the specified parameters.
 * @param load: Specifies the free-running downcounter value.
 * @param interrupt: Enable or disable interrupt.
 * @retval None
 */
void iwdt_init(uint32_t load, type_func_t interrupt)
{
	assert_param(IS_FUNC_STATE(interrupt));

	IWDT_UNLOCK();
	IWDT->LOAD.Word = load;
	IWDT->CON.CLKS  = 1;
	IWDT->CON.RSTEN = ENABLE;
	IWDT->CON.IE    = interrupt;
	IWDT_LOCK();

	return;
}

/**
 * @brief Start the IWDT
 * @retval None
 */
void iwdt_start(void)
{
	IWDT_UNLOCK();
	IWDT->CON.EN = 1;
	IWDT_LOCK();

	return;
}

/**
 * @brief Get the free-running downcounter value
 * @retval Value
 */
uint32_t iwdt_get_value(void)
{
	return IWDT->VALUE.Word;
}

/**
 * @brief Get interrupt state
 * @retval Value
 */
it_status_t iwdt_get_flag_status(void)
{
	return (it_status_t)(IWDT->RIS.WDTIF);
}

/**
 * @brief Refreshes the WWDT
 * @retval None
 */
void iwdt_feed_dog(void)
{
	IWDT_UNLOCK();
	IWDT->INTCLR.INTCLR = 0x1;
	IWDT_LOCK();

	return;
}
/**
 * @}
 */

#endif /* HAL_WDT */
/**
 * @}
 */

/**
 * @}
 */
